<?php
require_once 'abstract.php';


/**
 * Class VerifonePaymentCallbackModuleFrontController
 */
class VerifonePaymentCallbackModuleFrontController extends VerifoneAbstarctModuleFrontController
{

    /* @var $method */
    protected $method;

    public $errors = [];

    public $currentOrder;

    public $order;

    private $logger;

    public function init()
    {
        parent::init();
        $this->request = Tools::getAllValues();

    }

    /**
     * @throws PrestaShopException
     */
    public function postProcess()
    {
        $this->logger = new Verifone_Cart_Logger();
        $params = $this->request;

        try {
            $module_id = $params['id_module'];
            $cart_id = $params['id_cart'];

            if (!$module_id || !$cart_id) {
                throw new PrestaShopException($this->l('Verifone Payment Callback - No module or cart ID found.'));
            }

            $this->module = Module::getInstanceById($module_id);

            if (!isset($params['checkout_id'])) {
                throw new PrestaShopException(sprintf(
                    $this->l('Invalid callback parameters. Missing checkout_id. Params: %s'),
                    json_encode($params)
                ));
            }

            $cart = new Cart($cart_id);
            $customer = new Customer($cart->id_customer);

            $response = $this->module->client->getCheckout($params['checkout_id']);

            if (!isset($response['status'], $response['merchant_reference'], $response['transaction_id'])) {
                throw new PrestaShopException($this->l('Invalid API response: ') . json_encode($response));
            }

            $currency = Currency::getCurrencyInstance(Currency::getIdByIsoCode($response['currency_code']));
            $amount = $this->transformFromCents($response['amount'], $currency->precision);

            $shouldCreateOrder = ($response['status'] === "COMPLETED") ? true : false;

            $this->order = Order::getByCartId($response['merchant_reference']);

            if (!$this->order && $shouldCreateOrder) {
                $this->order = $this->validateVerifoneOrder(
                    $cart->id,
                    Configuration::get('PS_OS_PREPARATION'),
                    $amount,
                    $this->name,
                    'Order Created',
                    [],
                    null,
                    false,
                    $params['key'],
                    $currency->precision
                );
            }

            if (!$shouldCreateOrder) {
                $this->logger->error(sprintf(
                    $this->l('Payment declined or failed. Response: %s'),
                    json_encode($response)
                ));

                $this->context->cookie->__set('redirect_error', $this->l('Your payment could not be processed. Please try again.'));
                $this->context->cookie->write();

                Tools::redirect($this->context->link->getPageLink('order', true));
                exit;
            }

            if (!$this->order->hasInvoice()) {
                $this->order->setInvoice(true);
            }

            $this->redirectUrl = 'index.php?controller=order-confirmation&id_cart=' . $cart->id .
                '&id_module=' . $this->module->id .
                '&id_order=' . $this->order->id .
                '&key=' . $customer->secure_key;

            $history = new OrderHistory();
            $history->id_order = (int)$this->order->id;

            if (!$this->order->current_state) {
                $history->changeIdOrderState((int)Configuration::get('PS_OS_PREPARATION'), $this->order, true);
                $history->save();
            }

            foreach ($response['events'] as $event) {
                if ($event['type'] === 'TRANSACTION_SUCCESS') {
                    $orderPayments = OrderPayment::getByOrderReference($this->order->reference);

                    if (  Configuration::get('VERIFONE_ORDER_STATUS') != $this->order->current_state &&
                        Configuration::get('VERIFONE_TRANSACTION_TYPE') === 'SALE'
                    ) {
                        $history->changeIdOrderState((int)Configuration::get('VERIFONE_ORDER_STATUS'), $this->order, true);
                        $history->save();
                    }

                    foreach ($orderPayments as $payment) {
                        if ($payment->getOrderInvoice($this->order->id) && !$payment->transaction_id) {
                            $payment->transaction_id = $response['transaction_id'];
                            $payment->save();
                            break;
                        }
                    }
                }
            }

            $this->order->save();
            $this->cleanDuplicates($this->order->id);
            $this->cleanDuplicatePayments($this->order->reference);

        } catch (Exception $e) {
            $this->logger->error(sprintf($this->l('Exception during payment callback: %s'), $e->getMessage()));
            $this->context->cookie->__set('redirect_error', $this->l('Transaction was declined or failed!'));
            $this->context->cookie->write();

            Tools::redirect($this->context->link->getPageLink('cart', null, null, ['action' => 'show']));
            exit;
        }
    }


    /**
     * Sometimes, multiple updates of order history are created (no logic...) so we need to make some clean up
     * @param $id
     * @return void
     * @throws PrestaShopDatabaseException
     */
    private function cleanDuplicates($id)
    {
        $query = Db::getInstance()->query('SELECT id_order_history FROM ' . _DB_PREFIX_ . 'order_history WHERE id_order = ' . $id . ' AND id_order_state = ' . (int)Configuration::get('VERIFONE_ORDER_STATUS') . ' ORDER BY id_order_history ASC');
        $results = $query->fetchAll();
        foreach ($results as $k => $v) {
            if ($k > 0) { // keep the first history
                Db::getInstance()->execute('DELETE FROM ' . _DB_PREFIX_ . 'order_history WHERE id_order_history = ' . $v['id_order_history']);
            }
        }
    }

    /**
     * Sometimes, multiple payments are registered (no logic...) so we need to make some clean up
     * @param $reference
     * @return void
     * @throws PrestaShopDatabaseException
     */
    private function cleanDuplicatePayments($reference)
    {
        $query = Db::getInstance()->query('SELECT id_order_payment FROM ' . _DB_PREFIX_ . 'order_payment WHERE order_reference = "' . $reference . '" AND BINARY payment_method = "Verifone"');
        $results = $query->fetchAll();
        foreach ($results as $k => $v) {
            if ($k > 0) { // keep the first history
                Db::getInstance()->execute('DELETE FROM ' . _DB_PREFIX_ . 'order_payment WHERE id_order_payment = ' . $v['id_order_payment']);
            }
        }
    }

    /**
     * @throws PrestaShopException
     * @throws PrestaShopDatabaseException
     */
    public function validateVerifoneOrder(
        $id_cart,
        $id_order_state,
        $amount_paid,
        $payment_method = 'Unknown',
        $message = null,
        $transaction = [],
        $currency_special = null,
        $dont_touch_amount = false,
        $secure_key = false,
        $precision = 2
    ): Order
    {
        if ($this->needConvert()) {
            $amount_paid_curr = Tools::ps_round(Tools::convertPrice($amount_paid, new Currency($currency_special)), $precision);
        } else {
            $amount_paid_curr = Tools::ps_round($amount_paid, $precision);
        }

        $cart = new Cart((int)$id_cart);
        $total_ps = (float)$cart->getOrderTotal();
        if ($amount_paid_curr > $total_ps + 0.10 || $amount_paid_curr < $total_ps - 0.10) {
            $total_ps = $amount_paid_curr;
        }

        try {
            $this->module->validateOrder(
                (int)$id_cart,
                (int)$id_order_state,
                $total_ps,
                $payment_method,
                $message,
                ['transaction_id' => $transaction['transaction_id'] ?? ''],
                $currency_special,
                $dont_touch_amount,
                $secure_key
            );
        } catch (Exception $e) {


            $msg = sprintf($this->l('Order validation error : %s; File: %s; Line: %s;'), $e->getMessage(),
                $e->getFile(), $e->getLine());
            $this->logger->error($msg, __LINE__);

            $this->currentOrder = (int)Order::getIdByCartId((int)$id_cart);

            if (!$this->currentOrder) {
                $msg = sprintf($this->l('Order validation error : %s'), $e->getMessage());
            }
            throw new PrestaShopException($msg);
        }

        $order = new Order($this->module->currentOrder);

        if (isset($amount_paid_curr) && $amount_paid_curr != 0 && $order->total_paid != $amount_paid_curr && $this->isOneOrder($order->reference)) {
            $order->total_paid = $amount_paid_curr;
            $order->total_paid_real = $amount_paid_curr;
            $order->total_paid_tax_incl = $amount_paid_curr;
            $order->update();

            $sql = 'UPDATE `' . _DB_PREFIX_ . 'order_payment`
		    SET `amount` = ' . $amount_paid_curr . '
		    WHERE  `order_reference` = "' . pSQL($order->reference) . '"';
            Db::getInstance()->execute($sql);
        }

        return $order;
    }

    /**
     * Check if we need convert currency
     * @return boolean|integer currency id
     */
    public function needConvert()
    {
        $currency_mode = Currency::getPaymentCurrenciesSpecial($this->module->id);
        $mode_id = $currency_mode['id_currency'];
        if ($mode_id == -2) {
            return (int)Configuration::get('PS_CURRENCY_DEFAULT');
        } elseif ($mode_id == -1) {
            return false;
        } elseif ($mode_id != $this->context->currency->id) {
            return (int)$mode_id;
        } else {
            return false;
        }
    }

    public function isOneOrder($order_reference): bool
    {
        $query = new DBQuery();
        $query->select('COUNT(*)');
        $query->from('orders');
        $query->where('reference = "' . pSQL($order_reference) . '"');
        $countOrders = (int)DB::getInstance()->getValue($query);

        return $countOrders > 0;
    }

}
