<?php
require_once 'constants.php';
require_once 'abstract.php';


/**
 * Class VerifonePaymentCallbackModuleFrontController
 */
class VerifoneWebhookModuleFrontController extends VerifoneAbstarctModuleFrontController
{
    const TRANSACTION_STATUS_COMPLETED = 'COMPLETED';
    const TRANSACTION_STATUS_AUTHORISED = 'AUTHORISED';
    const TRANSACTION_STATUS_AUTHORIZED = 'AUTHORIZED';
    const TRANSACTION_STATUS_SALE_AUTHORISED = 'SALE AUTHORISED';
    const TRANSACTION_STATUS_SALE_SETTLED = 'SALE SETTLED';
    const TRANSACTION_STATUS_AUTHORISATION_AUTHORISED = 'AUTHORISATION AUTHORISED';

    const  TRANSACTION_STATUS_OK = [
        self::TRANSACTION_STATUS_COMPLETED,
        self::TRANSACTION_STATUS_AUTHORISED,
        self::TRANSACTION_STATUS_AUTHORIZED,
        self::TRANSACTION_STATUS_SALE_AUTHORISED,
        self::TRANSACTION_STATUS_SALE_SETTLED,
        self::TRANSACTION_STATUS_AUTHORISATION_AUTHORISED,
    ];

    public $errors;

    public $order;
    public $logger;

    public function init()
    {
        parent::init();
        $this->request = Tools::getAllValues();
        $this->logger = new Verifone_Cart_Logger();
        
    }


    /**
     * @return string
     * @throws Exception
     */
    public function postProcess(): string
    {

        $input = file_get_contents('php://input');
        if (!$input) {
            $this->logger->error('Unable to process data');
            throw new PrestaShopException('Unable to process data');
        }
        $data = json_decode($input, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->logger->error(__CLASS__ . '@' . __FUNCTION__ . ' line ' . __LINE__ . ': Unable to decode response from API');
            throw new PrestaShopException(__CLASS__ . '@' . __FUNCTION__ . ' line ' . __LINE__ . ': Unable to decode response from API');
        }

        if (strpos($data['eventType'], 'Checkout') !== false) {
            try {
                $checkout = $this->module->client->getCheckout($data['recordId']);
                $data['recordId'] = $checkout['transaction_id'];
            } catch (PrestaShopException $exception) {
                $this->logger->error(__CLASS__ . '@' . __FUNCTION__ . ' line ' . __LINE__ . ': Unable to decode response from API');
                throw new PrestaShopException($exception->getMessage());
            }
        }
        try {
            $response = $this->module->client->getTransaction($data['recordId']);
        } catch (PrestaShopException $exception) {
            $this->logger->error(__CLASS__ . '@' . __FUNCTION__ . ' line ' . __LINE__ . ': There was a request for an order: ' . $exception->getMessage());
            throw new PrestaShopException('Unable to process data');
        }
        $order = Order::getByCartId($response['merchant_reference']);
        if (!$order) {
            $this->createOrderBasedOnCartId($response, $data['eventType']);
        } else {
            switch ($response['status']) {
                case VERIFONE_TRANSACTION_CAPTURE_AUTHORISED:
                case VERIFONE_TRANSACTION_CAPTURE_SETTLED:
                    //if we have a capture, then we update order status
                    if ($response['transaction_type'] === 'CAPTURE' &&
                        (int)Configuration::get('VERIFONE_ORDER_STATUS') != $order->current_state) {
                        $history = new OrderHistory();
                        $history->id_order = (int)$order->id;
                        $history->changeIdOrderState((int)Configuration::get('VERIFONE_ORDER_STATUS'), (int)$order->id, true);
                        $history->save();
                    }
                    break;
                case VERIFONE_TRANSACTION_AUTHORISED:
                case VERIFONE_TRANSACTION_SALE_AUTHORISED:
                case VERIFONE_TRANSACTION_SALE_SETTLED:
                case VERIFONE_TRANSACTION_AUTHORISATION_AUTHORISED:
                case VERIFONE_TRANSACTION_PREAUTH_AUTHORISED:
                    break;
                case VERIFONE_TRANSACTION_PENDING:
                    $orderNote = 'The payment is not done yet (transaction is in PRE/FINAL AUTH status), use the CAPTURE button to fully/partial capture the amount.
                    Attempting to refund the order in part or in full, before capturing, will void the authorization and cancel the payment.';
                    $this->addOrderNote($order, $orderNote);
                    break;
                case VERIFONE_TRANSACTION_REFUND_AUTHORISED:
                case VERIFONE_TRANSACTION_REFUND_REFUNDED:
                case VERIFONE_TRANSACTION_REFUNDED:
                case VERIFONE_TRANSACTION_REFUND_SETTLED:
                    //we add this order not only for refunds, but we don't change order status
                    $orderNote = 'Order status changed to: ' . $this->getOrderState((int)Configuration::get('PS_OS_REFUND'))['name'] . '. Amount refunded: ' . $response['amount'] . $response['currency_code'];
                    $this->addOrderNote($order, $orderNote);
                    break;
                default:
                    $orderNote = null;
                    if (VERIFONE_WEBHOOK_EVENT_TYPE_REFUND_DECLINED === $data['eventType']) {
                        $orderNote = 'Order payment refund failed';
                    }
                    if (VERIFONE_WEBHOOK_EVENT_3DS_AUTH_SUCCESS !== $data['eventType']) {
                        $orderNote = 'Order status changed to: ' . $this->getOrderState((int)Configuration::get('PS_OS_ERROR'))['name'];
                    }
                    if ($orderNote) {
                        $this->addOrderNote($order, $orderNote);
                    }
            }
            $this->addOrderNote($order, 'Verifone notification event: ' . $data['eventType']);

        }

        return json_encode(['success' => true]);
    }


    public function createOrderBasedOnCartId($response, $eventType)
    {
 
        try {
            $cart = new Cart($response['merchant_reference']);
            $payment_currency_id = Currency::getIdByIsoCode($response['currency_code']);
            $payment_currency = Currency::getCurrencyInstance($payment_currency_id);
            //Now generate Order!
            $this->order = $this->validateVerifoneOrder(
                $cart->id,
                $response['amount'],
                $payment_currency->precision,
                $response['id']
            );
            $this->addOrderNote($this->order, 'Verifone notification event: ' . $eventType);
            if (!$this->order->current_state) {
                $history = new OrderHistory();
                $history->id_order = (int)$this->order->id;
                $history->changeIdOrderState((int)Configuration::get('PS_OS_PREPARATION'), $this->order, true);
                $history->save();
                $this->logger->log(sprintf($this->l('Changing order "%s" status to "Processing"'), $this->order->reference));
            }
            // set the order status based on the checkout response
            if (in_array($response['status'], self::TRANSACTION_STATUS_OK)) {
                $orderPayment = OrderPayment::getByOrderReference($this->order->reference);
                $current = $this->getCurrentOrderState($this->order->id);
                foreach ($orderPayment as $payment) {
                    $invoice = $payment->getOrderInvoice($this->order->id);
                    if ($invoice && !$payment->transaction_id) {
                        $payment->transaction_id = $response['id'];
                        $payment->save();
                    }
                    if (!$current || (Configuration::get('VERIFONE_ORDER_STATUS') != $current)) {
                        // if the transaction needs to be captured, we don't update the status yet
                        if (!in_array($response['status'], ['AUTHORISATION AUTHORISED', 'AUTHORISED', 'CAPTURE AUTHORISED'])
                            && in_array($response['transaction_type'], ['SALE', 'AUTHORISATION'])) {
                            $history = new OrderHistory();
                            $history->id_order = (int)$this->order->id;
                            $history->changeIdOrderState((int)Configuration::get('VERIFONE_ORDER_STATUS'), $this->order, true);
                            $history->save();
                        }
                    }
                }
            }
            if (strpos($response['status'], 'FAILED') !== false) {
                $logger_msg = $this->l('Payment FAILED for order: ' . $this->order->reference . '. Received response: ' . json_encode($response));
                $this->logger->error($logger_msg);
                $history = new OrderHistory();
                $history->id_order = (int)$this->order->id;
                $history->changeIdOrderState((int)Configuration::get('PS_OS_ERROR'), $this->order, true);
                $history->save();
                $msg = sprintf($this->l('Payment FAILED for order: %s.'), $this->order->reference);
                throw new PrestaShopException($msg);
            }

            if (strpos($response['status'], 'DECLINED') !== false) {
                $logger_msg = sprintf($this->l('Payment DECLINED for order: %s. Received response: %s '),
                    $this->order->reference, json_encode($response));
                $this->logger->error($logger_msg);
                $history = new OrderHistory();
                $history->id_order = (int)$this->order->id;
                $history->changeIdOrderState((int)Configuration::get('PS_OS_ERROR'), $this->order, true);
                $history->save();
                $msg = sprintf($this->l('Payment DECLINED for order: %s.'), $this->order->reference);
                throw new PrestaShopException($msg);
            }

            $this->order->save();

        } catch (Exception $e) {
            $this->errors['err'] = sprintf($this->l('Payment failed for order %s, please try again or contact the merchant!'), $this->order->reference);
            $this->logger->error(sprintf($this->l('Exception handling payment callback: %s'),
                $e->getMessage()));
        }
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    private function validateVerifoneOrder($id_cart, $amount_paid, $precision = 2, $transaction_id = null)
    {
        if ($this->needConvert()) {
            $amount_paid_curr = Tools::ps_round(Tools::convertPrice($amount_paid, new Currency()), $precision);
        } else {
            $amount_paid_curr = Tools::ps_round($amount_paid, $precision);
        }

        $cart = new Cart((int)$id_cart);
        $total_ps = (float)$cart->getOrderTotal();
        if ($amount_paid_curr > $total_ps + 0.10 || $amount_paid_curr < $total_ps - 0.10) {
            $total_ps = $amount_paid_curr;
        }

        try {
            $this->module->validateOrder(
                (int)$id_cart,
                (int)Configuration::get('PS_OS_PREPARATION'),
                $total_ps,
                $this->name,
                'Order created',
                ['transaction_id' => $transaction_id ?? '']
            );
        } catch (Exception $e) {

            $msg = sprintf($this->l('Order validation error : "%s"; File: "%s"; Line: "%s";'), $e->getMessage(),
                $e->getFile(), $e->getLine());
            $this->logger->error($msg);

            $this->currentOrder = (int)Order::getIdByCartId((int)$id_cart);

            if (!$this->currentOrder) {
                $msg = sprintf($this->l('Order validation error : "%s"'), $e->getMessage());
            }
            throw new PrestaShopException($msg);
        }

        $order = new Order($this->module->currentOrder);

        if (isset($amount_paid_curr) && $amount_paid_curr != 0 && $order->total_paid != $amount_paid_curr && $this->isOneOrder($order->reference)) {
            $order->total_paid = $amount_paid_curr;
            $order->total_paid_real = $amount_paid_curr;
            $order->total_paid_tax_incl = $amount_paid_curr;
            $order->update();

            $sql = 'UPDATE `' . _DB_PREFIX_ . 'order_payment`
		    SET `amount` = ' . $amount_paid_curr . '
		    WHERE  `order_reference` = "' . pSQL($order->reference) . '"';
            Db::getInstance()->execute($sql);
        }

        return $order;
    }

    /**
     * Check if we need convert currency
     * @return boolean|integer currency id
     */
    private function needConvert()
    {
        $currency_mode = Currency::getPaymentCurrenciesSpecial($this->module->id);
        $mode_id = $currency_mode['id_currency'];
        if ($mode_id == -2) {
            return (int)Configuration::get('PS_CURRENCY_DEFAULT');
        } elseif ($mode_id != $this->context->currency->id) {
            return (int)$mode_id;
        }

        return false;
    }

    private function isOneOrder($order_reference)
    {
        $query = new DBQuery();
        $query->select('COUNT(*)');
        $query->from('orders');
        $query->where('reference = "' . pSQL($order_reference) . '"');
        $countOrders = (int)DB::getInstance()->getValue($query);

        return $countOrders == 1;
    }

    private function getOrderState($state)
    {
        $query = new DBQuery();
        $query->select('*');
        $query->from('order_state_lang');
        $query->where('id_order_state = "' . pSQL($state) . '"');
        $query->where('id_lang = "' . $this->context->language->id . '"');

        return DB::getInstance()->getRow($query);
    }

    private function getCurrentOrderState($id)
    {
        $currentOrderState = Db::getInstance()->getRow('SELECT * FROM ' . _DB_PREFIX_ . 'order_history WHERE id_order = ' . $id . ' ORDER BY id_order_history DESC');

        return $currentOrderState ? $currentOrderState['id_order_state'] : null;
    }
}
