<?php

namespace VerifoneEcomAPI\ApiWrapper\Http\ApiClient;

use Configuration;
use Exception;
use VerifoneEcomAPI\ApiWrapper\Authentication\Interfaces\AuthenticationInterface;
use VerifoneEcomAPI\ApiWrapper\Http\HttpException;
use VerifoneEcomAPI\ApiWrapper\Http\SimpleCurl;
use VerifoneEcomAPI\ApiWrapper\Regions\AbstractRegion;
use VerifoneEcomAPI\ApiWrapper\Schemas\SchemaInterface;
use VerifoneEcomAPI\ApiWrapper\Settings;
use VerifoneEcomAPI\ApiWrapper\Validators\Validator;

/**
 * Class Client
 * @package VerifoneEcomAPI\ApiWrapper\Http\ApiClient
 */
class Client
{

    private $region;
    private $simpleCurl;
    private $settings;
    private $authentication;
    private $auth;

    /**
     * Client constructor.
     * @param AbstractRegion $region
     * @param Settings $settings
     * @param AuthenticationInterface $authentication
     * @param SimpleCurl $simpleCurl
     */
    public function __construct(
        AbstractRegion          $region,
        Settings                $settings,
        AuthenticationInterface $authentication,
        SimpleCurl              $simpleCurl
    )
    {
        $this->region = $region;
        $this->authentication = $authentication;
        $this->settings = $settings;
        $this->simpleCurl = $simpleCurl;
        $this->auth = $this->authentication->getAuth();
    }

    /**
     * @param array $payload
     * @param string $transactionId
     * @param string|null $type
     * @return mixed
     * @throws HttpException
     */
    public function postCapture(array $payload, string $transactionId, string $type = null)
    {

        $result = $this->simpleCurl->setOpt(CURLOPT_URL,
            $type === 'PayPal'
                ? $this->region->postPaypalCaptureUrl($transactionId)
                : $this->region->postCaptureUrl($transactionId)
        )
            ->setOpt(CURLOPT_RETURNTRANSFER, true)
            ->setOpt(CURLOPT_POST, true)
            ->setOpt(CURLOPT_POSTFIELDS, json_encode($payload))
            ->setOpt(CURLOPT_HTTPHEADER, $this->getHeaders());
        $this->addSsl($result);

        return $this->decodeResult($result->getResult());
    }

    /**
     * @param array $payload
     * @param $transactionId
     * @return mixed
     * @throws HttpException
     */
    public function postVoidAuthorization(array $payload, $transactionId)
    {

        $result = $this->simpleCurl->setOpt(CURLOPT_URL, $this->region->postVoidAuthorizationUrl($transactionId))
            ->setOpt(CURLOPT_RETURNTRANSFER, true)
            ->setOpt(CURLOPT_POST, true)
            ->setOpt(CURLOPT_POSTFIELDS, json_encode($payload))
            ->setOpt(CURLOPT_HTTPHEADER, $this->getHeaders());
        $this->addSsl($result);

        return $this->decodeResult($result->getResult());
    }

    /**
     * @param array $payload
     * @return mixed
     * @throws HttpException
     */
    public function postCustomer(array $payload)
    {
        $result = $this->simpleCurl->setOpt(CURLOPT_URL, $this->region->postCustomerUrl())
            ->setOpt(CURLOPT_RETURNTRANSFER, true)
            ->setOpt(CURLOPT_POST, true)
            ->setOpt(CURLOPT_POSTFIELDS, json_encode($payload))
            ->setOpt(CURLOPT_HTTPHEADER, $this->getHeaders());
        $this->addSsl($result);

        return $this->decodeResult($result->getResult());
    }

    /**
     * @param array $payload
     * @return mixed
     * @throws HttpException
     */
    public function postCheckout(array $payload)
    {
        $result = $this->simpleCurl->setOpt(CURLOPT_URL, $this->region->postCheckoutUrl())
            ->setOpt(CURLOPT_RETURNTRANSFER, true)
            ->setOpt(CURLOPT_POST, true)
            ->setOpt(CURLOPT_POSTFIELDS, json_encode($payload))
            ->setOpt(CURLOPT_HTTPHEADER, $this->getHeaders());
        $this->addSsl($result);

        return $this->decodeResult($result->getResult());
    }

    /**
     * @param array $payload
     * @param $transactionId
     * @return mixed
     * @throws HttpException
     */
    public function postRefund(array $payload, $transactionId)
    {

        $result = $this->simpleCurl->setOpt(CURLOPT_URL, $this->region->getRefundUrl($transactionId))
            ->setOpt(CURLOPT_RETURNTRANSFER, true)
            ->setOpt(CURLOPT_POST, true)
            ->setOpt(CURLOPT_POSTFIELDS, json_encode($payload))
            ->setOpt(CURLOPT_HTTPHEADER, $this->getHeaders());
        $this->addSsl($result);

        return $this->decodeResult($result->getResult());
    }

    /**
     * @param $checkoutId
     * @return mixed
     * @throws HttpException
     */
    public function getCheckout($checkoutId)
    {
        $result = $this->simpleCurl->setOpt(CURLOPT_URL, $this->region->getCheckoutUrl($checkoutId))
            ->setOpt(CURLOPT_RETURNTRANSFER, true)
            ->setOpt(CURLOPT_HTTPHEADER, $this->getHeaders());
        $this->addSsl($result);

        return $this->decodeResult($result->getResult());
    }

    /**
     * @param $client
     */
    private function addSsl($client)
    {
        if (!$this->settings->getCurlVerifySsl()) {
            $client->setOpt(CURLOPT_SSL_VERIFYHOST, false)
                ->setOpt(CURLOPT_SSL_VERIFYPEER, false);
        }
    }

    /**
     * @return string[]
     */
    private function getHeaders()
    {

        return [
            'Content-Type: application/json',
            'Authorization: Basic ' . $this->auth,
        ];
    }

    /**
     * @param $result
     * @return mixed
     * @throws HttpException
     */
    private function decodeResult($result)
    {
        $result = json_decode($result, true);
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new HttpException('Unable to decode response from API');
        }

        return $result;
    }

    /**
     * @param $transactionId
     *
     * @return mixed
     * @throws HttpException
     */
    public function getTransaction($transactionId)
    {
        $result = $this->simpleCurl->setOpt(CURLOPT_URL, $this->region->getTransactionUrl($transactionId))
            ->setOpt(CURLOPT_RETURNTRANSFER, true)
            ->setOpt(CURLOPT_HTTPHEADER, $this->getHeaders());

        $this->addSsl($result);

        return $this->decodeResult($result->getResult());
    }

    /**
     * Start auto setup methods
     */

    /**
     * @param string $path
     * @param array $params
     *
     * @return array|bool|float|int|object|string|null
     * @throws HttpException
     */
    protected function callEntityService( string $path, array $params = [] ) {
        $extraParams = array_merge( [
            'start'                => 0,
            'limit'                => 100,
            'order'                => 'ASC',
            'populateParentEntity' => true,
            'status'               => 'ACTIVE'
        ], $params );


        $result = $this->simpleCurl
            ->setOpt( CURLOPT_URL,
                $this->region->getEntityServicePortalUrl( $path . '?' . http_build_query( $extraParams ) ) )
            ->setOpt( CURLOPT_RETURNTRANSFER,
                true )
            ->setOpt( CURLOPT_HTTPHEADER,
                $this->getHeaders() );

        $this->addSsl( $result );

        return $this->decodeResult( $result->getResult() );
    }

    /**
     * @return array|bool|float|int|object|string|null
     * @throws HttpException
     */
    public function getEntities() {

        return $this->callEntityService( 'entities' );
    }

    /**
     * @param string $entityId
     *
     * @return float|object|array|bool|int|string|null
     * @throws HttpException
     */
    public function getPPCs( string $entityId ) {

        return $this->callEntityService( 'paymentContracts', [ 'parentIds' => $entityId ] );
    }

    /**
     * @return float|object|array|bool|int|string|null
     * @throws HttpException
     */
    public function get3DS() {

        return $this->callEntityService( 'threeDSContracts' );
    }

    /**
     * @return float|object|array|bool|int|string|null
     * @throws HttpException
     */
    public function getWallets() {

        return $this->callEntityService( 'walletContracts' );
    }

    public function checkCredentials(): bool {
        try {
            $result = $this->simpleCurl
                ->setOpt( CURLOPT_URL, $this->region->getTransactionCountURL() )
                ->setOpt( CURLOPT_RETURNTRANSFER, true )
                ->setOpt( CURLOPT_HTTPHEADER, $this->getHeaders() );
            $this->addSsl( $result );

            $result = $this->decodeResult( $result->getResult() );

            return isset($result['count']);
        } catch ( Exception $exception ) {
            return false;
        }
    }

    public function getWebhooks() {
        $path   = '?eventTypes=CheckoutTransactionSuccess&order=DESC&orderBy=createdDate&pageNumber=1&pageSize=20&status=ACTIVE';
        $result = $this->simpleCurl->setOpt( CURLOPT_URL, $this->region->getWebhooksUrl( $path ) )
            ->setOpt( CURLOPT_RETURNTRANSFER, true )
            ->setOpt( CURLOPT_HTTPHEADER, $this->getHeaders() );
        $this->addSsl( $result );

        return $this->decodeResult( $result->getResult() );
    }

    public function setWebhook( $entityId, $url ) {
        $params = [
            'name'               => Configuration::get('PS_SHOP_NAME'),
            'eventTypes'         => [ 'CheckoutTransactionSuccess' ],
            'entities'           => [ $entityId ],
            'emailNotifications' => [],
            'endpoints'          => [ [ 'url' => $url ] ]
        ];
        $result = $this->simpleCurl->setOpt( CURLOPT_URL, $this->region->getWebhooksUrl() )
            ->setOpt( CURLOPT_RETURNTRANSFER, true )
            ->setOpt( CURLOPT_POST, true )
            ->setOpt( CURLOPT_POSTFIELDS, json_encode( $params ) )
            ->setOpt( CURLOPT_HTTPHEADER, $this->getHeaders() );
        $this->addSsl( $result );

        return $this->decodeResult( $result->getResult() );
    }

    /**
     * end auto setup methods
     */
}