<?php

namespace VerifoneEcomAPI\ApiWrapper\Http;

use Configuration;
use Context;
use Exception;
use VerifoneEcomAPI\ApiWrapper\Authentication\BasicAuth;
use VerifoneEcomAPI\ApiWrapper\Authentication\Interfaces\AuthenticationInterface;
use VerifoneEcomAPI\ApiWrapper\Http\ApiClient\Client;
use VerifoneEcomAPI\ApiWrapper\Regions\AbstractRegion;
use VerifoneEcomAPI\ApiWrapper\Regions\Au;
use VerifoneEcomAPI\ApiWrapper\Regions\Emea;
use VerifoneEcomAPI\ApiWrapper\Regions\Na;
use VerifoneEcomAPI\ApiWrapper\Regions\NzAu;
use VerifoneEcomAPI\ApiWrapper\Settings;

class AutoSetupService
{
    private Client $client;
    private Settings $settings;
    private BasicAuth $auth;
    private array $cardList = ['AMEX', 'CB', 'VPAY', 'ISRAEL_PRIVATE_LABEL', 'ISRACARD', 'Q_CARD', 'CARNET', 'DINERS', 'DISCOVER', 'ELECTRON', 'JCB', 'MAESTRO', 'UPI', 'VISA', 'MASTERCARD'];
    private string $userId;
    private string $apiKey;
    private bool $test;
    private string $region;
    protected array $allFields = [
        'VERIFONE_CHECKOUT_TYPE',
        'VERIFONE_THEME_ID',
        'VERIFONE_CARD_ENABLED',
        'VERIFONE_PAYMENT_CONTRACT_ID',
        'VERIFONE_3DS_ENABLED',
        'VERIFONE_3DS_CONTRACT_ID',
        'VERIFONE_PAYPAL_ENABLED',
        'VERIFONE_PAYPAL_CONTRACT_ID',
        'VERIFONE_APPLE_PAY_ENABLED',
        'VERIFONE_APPLE_PAY_CONTRACT_ID',
        'VERIFONE_APPLE_PAY_THREE_DS_CONTRACT_ID',
        'VERIFONE_APPLE_PAY_SCA_LEVEL',
        'VERIFONE_GOOGLE_PAY_ENABLED',
        'VERIFONE_GOOGLE_PAY_CONTRACT_ID',
        'VERIFONE_GOOGLE_PAY_THREE_DS_CONTRACT_ID',
        'VERIFONE_GOOGLE_PAY_SCA_LEVEL',
        'VERIFONE_KLARNA_ENABLED',
        'VERIFONE_CURL_VERIFY_SSL',
        'VERIFONE_CARD_FRAUD_PROTECTION_ENABLED',
        'VERIFONE_CARD_FRAUD_PROTECTION_CONTRACT_ID',
        'VERIFONE_SWISH_ENABLED',
        'VERIFONE_VIPPS_ENABLED',
        'VERIFONE_VIPPS_CONTRACT_ID',
        'VERIFONE_VIPPS_THREE_DS_CONTRACT_ID',
        'VERIFONE_VIPPS_SCA_LEVEL',
        'VERIFONE_GPP2_ENABLED',
        'VERIFONE_GPP2_CONTRACT_ID',
        'VERIFONE_MOBILE_PAY_ENABLED',
        'VERIFONE_MOBILE_PAY_CONTRACT_ID',
        'VERIFONE_MOBILE_PAY_THREE_DS_CONTRACT_ID',
        'VERIFONE_MOBILE_PAY_SCA_LEVEL',
        'VERIFONE_ABS_INSTALLMENTS_ENABLED',
        'VERIFONE_ABS_INSTALLMENTS',
        'VERIFONE_GIFT_CARD_ENABLED',
        'VERIFONE_GIFT_CARD_CONTRACT_ID',
        'VERIFONE_PLCC_ENABLED',
        'VERIFONE_PLCC_CONTRACT_ID',
        'VERIFONE_BANK_OP_ENABLED',
        'VERIFONE_BANK_OP_CONTRACT_ID'
    ];

    public function __construct(array $params)
    {
        $this->userId = $params['user_id'];
        $this->apiKey = $params['api_key'];
        $this->test = $params['env'] !== 'live';
        $this->region = $params['region'];
        $this->settings = new Settings();
        $this->settings->set('testMode', $this->test)
            ->set('apiKey', $this->apiKey)
            ->set('userId', $this->userId)
            ->set('region', $this->region);

        $this->auth = new BasicAuth($this->settings);
        $objRegion = match ($params['region']) {
            'NZAU' => new NzAu($this->settings),
            'NA' => new Na($this->settings),
            'AU' => new Au($this->settings),
            default => new Emea($this->settings),
        };

        $this->client = new Client($objRegion, $this->settings, $this->auth, new SimpleCurl());
    }

    /**
     *  List all entities for this user_id-api_key pair
     *  User must have access only to one Merchant Company and 1 Merchant Site org, or just 1 Merchant Site org
     *  if there are more Merchant Company or Site org, we return an error
     * @return false|mixed
     * @throws HttpException
     */
    private function getEntities()
    {
        $entityList = $this->client->getEntities();
        $list = [];
        foreach ($entityList as $entity) {
            if ($entity['entityType'] === 'MERCHANT_SITE') {
                $list[] = $entity;
            }
        }

        return count($list) === 1 ? $list[0] : false;
    }


    /**
     * @throws HttpException
     */
    private function getPPCs($entityId)
    {
        return $this->client->getPPCs($entityId);
    }

    /**
     * User only has 1 PPC for Cards (might be separate PPCs for Gift Cards and PLCC)
     *
     * @param $ppcList
     *
     * @return array|bool|float|int|object|string|null
     */
    private function getPPC4card($ppcList)
    {
        $key = 0;
        $validPPC = null;
        foreach ($ppcList as $ppc) { // iterate through all PPCs
            foreach ($ppc['paymentType'] as $pt) { // we look in all payment types
                if (in_array($pt, $this->cardList) && in_array('ECOMMERCE', $ppc['salesChannels'])) { // check if it's a valid card name (not like PayPal or something else)
                    $key++;
                    $validPPC = $ppc;
                    break;
                }
            }
        }

        return $key === 1 ? $validPPC : null;
    }


    /**
     * User only has one PPC for Paypal
     *
     * @param $ppcList
     *
     * @return array|bool|float|int|object|string|null
     */
    private function getPaypalConfig($ppcList)
    {
        $key = 0;
        $validPPC = null;
        foreach ($ppcList as $ppc) { // iterate through all PPCs
            foreach ($ppc['paymentType'] as $pt) { // we look in all payment types
                if (in_array($pt, ['PAYPAL_ECOM', 'PAYPAL_ECOM_MANAGED']) && in_array('ECOMMERCE', $ppc['salesChannels'])) {
                    $key++;
                    $validPPC = $ppc;
                    break;
                }
            }
        }

        return $key === 1 ? $validPPC : null;
    }

    /**
     * @throws HttpException
     */
    private function get3DS()
    {
        $threeDs = $this->client->get3DS();

        return count($threeDs) ? $threeDs[0] : false;
    }


    /**
     * @throws HttpException
     */
    private function getWallets()
    {
        return $this->client->getWallets();
    }


    /**
     * @throws HttpException
     */
    public function setAccount(): array
    {
        if (!$this->client->checkCredentials()) {
            return [
                'status' => false,
                'message' => 'Invalid credentials. Make sure you enter right credentials/environment and try again!'
            ];
        }

        //default response
        $response = [
            'status' => false,
            'message' => 'Account could not be auto-configured, manual configuration is required'
        ];
        $entity = $this->getEntities();

        if (!$entity) {
            return $response;
        }
        $ppcList = $this->getPPCs($entity['entityUid']);
        $paypalPpc = $this->getPaypalConfig($ppcList);
        $ppc = $this->getPPC4card($ppcList);
        if (!$ppc && !$paypalPpc) {
            return $response;
        }


        $this->update('VERIFONE_MODE', $this->test);
        $this->update('VERIFONE_TITLE', 'Verifone Checkout');
        $this->update('VERIFONE_DESCRIPTION', 'Payments powered by Verifone');
        $this->update('VERIFONE_API_KEY', $this->apiKey);
        $this->update('VERIFONE_USER_ID', $this->userId);
        $this->update('VERIFONE_REGION', $this->region);
        $this->update('VERIFONE_ENTITY_ID', $entity['entityUid']);
        $this->update('VERIFONE_TRANSACTION_TYPE', 'SALE');
        $this->update('VERIFONE_ORDER_STATUS', '2'); //payment accepted
        $this->update('VERIFONE_CHECKOUT_TYPE', 'iframe'); // or null = same result

        if ($paypalPpc) {
            $this->update('VERIFONE_PAYPAL_ENABLED', 1);
            $this->update('VERIFONE_PAYPAL_CONTRACT_ID', $paypalPpc['contractUid']);
        }

        // card 3ds
        $threeDs = $this->get3DS();
        if ($threeDs) {
            $this->update('VERIFONE_3DS_ENABLED', 1);
            $this->update('VERIFONE_3DS_CONTRACT_ID', $threeDs['contractUid']);
        }

        //card
        if (array_diff($this->cardList, $ppc['paymentType'])) {
            $this->update('VERIFONE_CARD_ENABLED', 1);
            $this->update('VERIFONE_PAYMENT_CONTRACT_ID', $ppc['contractUid']);
        }
            //wallets
            $wallets = $this->getWallets();
            foreach ($wallets as $wallet) {
                if ($wallet['contractType'] === 'VIPPS_ECOM') { //vipps
                    $this->update('VERIFONE_VIPPS_ENABLED', 1);
                    $this->update('VERIFONE_VIPPS_CONTRACT_ID', $ppc['contractUid']);
                    $this->update('VERIFONE_VIPPS_SCA_LEVEL', $threeDs ? 'WALLET' : 'NONE');
                    if ($threeDs) {
                        $this->update('VERIFONE_VIPPS_THREE_DS_CONTRACT_ID', $threeDs['contractUid']);
                    }
                }
                if ($wallet['contractType'] === 'MOBILEPAY_ECOM') { //mobile pay
                    $this->update('VERIFONE_MOBILE_PAY_ENABLED', 1);
                    $this->update('VERIFONE_MOBILE_PAY_CONTRACT_ID', $ppc['contractUid']);
                    $this->update('VERIFONE_MOBILE_PAY_SCA_LEVEL', $threeDs ? 'WALLET' : 'NONE');
                    if ($threeDs) {
                        $this->update('VERIFONE_MOBILE_PAY_THREE_DS_CONTRACT_ID', $threeDs['contractUid']);
                    }
                }
                if ($wallet['contractType'] === 'APPLE_PAY_WEB') { //Apple Pay
                    $this->update('VERIFONE_APPLE_PAY_ENABLED', 1);
                    $this->update('VERIFONE_APPLE_PAY_CONTRACT_ID', $ppc['contractUid']);
                    $this->update('VERIFONE_APPLE_PAY_SCA_LEVEL', $threeDs ? 'WALLET' : 'NONE');
                    if ($threeDs) {
                        $this->update('VERIFONE_APPLE_PAY_THREE_DS_CONTRACT_ID', $threeDs['contractUid']);
                    }
                }

                if ($wallet['contractType'] === 'GOOGLE_PAY') { //google pay
                    $this->update('VERIFONE_GOOGLE_PAY_ENABLED', 1);
                    $this->update('VERIFONE_GOOGLE_PAY_CONTRACT_ID', $ppc['contractUid']);
                    $this->update('VERIFONE_GOOGLE_PAY_SCA_LEVEL', $threeDs ? 'WALLET' : 'NONE');
                    if ($threeDs) {
                        $this->update('VERIFONE_GOOGLE_PAY_THREE_DS_CONTRACT_ID', $threeDs['contractUid']);
                    }
                }
            }


        foreach ($ppcList as $item) {
            //klarna
            if (in_array('KLARNA_ECOM', $item['paymentType'])) {
                $this->update('VERIFONE_KLARNA_ENABLED', 1);
            }
            //swish
            if (in_array('SWISH', $item['paymentType'])) {
                $this->update('VERIFONE_SWISH_ENABLED', 1);
            }
            //gift card
            if (in_array('GIFT_CARD', $item['paymentType'])) {
                $this->update('VERIFONE_GIFT_CARD_ENABLED', 1);
                $this->update('VERIFONE_GIFT_CARD_CONTRACT_ID', $item['contractUid']);
            }
            //plcc
            if (in_array('PLCC', $item['paymentType'])) {
                $this->update('VERIFONE_PLCC_ENABLED', 1);
                $this->update('VERIFONE_PLCC_CONTRACT_ID', $item['contractUid']);
            }
            //bankOp
            if (in_array('OP_ONLINE_PAYMENT', $item['paymentType'])) {
            $this->update('VERIFONE_BANK_OP_ENABLED', 1);
            $this->update('VERIFONE_BANK_OP_CONTRACT_ID', $item['contractUid']);
            }
        }
        $this->webhookSetup($entity['entityUid']);

        //clear older/unnecessary settings
        foreach ($this->allFields as $field) {
            Configuration::updateValue($field, NULL);
        }

        return [
            'status' => true,
            'message' => 'Your account has been successfully configured!',
        ];
    }


    public function webhookSetup($entityId): void
    {
        $url = Context::getContext()->link->getModuleLink('verifone', 'webhook', [], true);
        $webhooks = $this->client->getWebhooks();
        // if we find a hook that matches our url, we stop
        foreach ($webhooks['triggers'] as $trigger) {
            foreach ($trigger['endpoints'] as $endpoint) {
                if ($endpoint['url'] === $url) {
                    return;
                }
            }
        }

        //save new webhook
        $this->client->setWebhook($entityId, $url);
    }

    private function update($key, $value)
    {

        //create/update new key=>value
        Configuration::updateValue($key, $value);

        //remove it from an array
        if ($index = array_search($key, $this->allFields)) {
            unset($this->allFields[$index]);
        }
    }
}
