<?php


use VerifoneEcomAPI\ApiWrapper\Settings;

/**
 * Class Verifone_Checkout
 */
class Verifone_Checkout extends Verifone_Api_Entity
{
    public $cart;
    private $customer_id;
    private $helper;
    private $settings;
    private $currencyIso;
    private $params;
    protected $data = [];
    protected $precision;
    protected $transactionType;

    /**
     * Verifone_Checkout constructor.
     *
     * @param $params
     * @param Settings $settings
     * @param Verifone_Helper $helper
     * @param string $customer_id
     *
     * @throws Exception
     */
    public function __construct($params, Settings $settings, Verifone_Helper $helper, ?string $customer_id = '')
    {
        $this->params = $params;
        $this->currencyIso = strtoupper($params['currency']->iso_code);
        $this->cart = $params['cart'];
        $this->customer_id = $customer_id;
        $this->helper = $helper;
        $this->settings = $settings;
        $this->precision = $this->getCurrencyPrecision($this->currencyIso);
        $this->transactionType = $this->helper->getDeviceType();
        $this->buildData();
    }

    /**
     * @throws Exception
     */
    private function buildData()
    {

        $this->data = [
            'customer' => $this->customer_id,
            'amount' => $this->helper->calculateAmount($this->cart->getOrderTotal(true, CartCore::BOTH), $this->currencyIso),
            'currency_code' => $this->currencyIso,
            'entity_id' => $this->settings->getEntityId(),
            'merchant_reference' => (string)$this->cart->id,
            'redirect_method' => 'HEADER_REDIRECT',
            'return_url' => $this->params['returnUrl'],
            'i18n' => $this->getLanguage(),
            'interaction_type' => strtoupper($this->settings->getCheckoutType()),
            'configurations' => []
        ];
        $this->card();
        $this->giftCard();
        $this->paypal();
        $this->theme();
        $this->applePay();
        $this->googlePay();
        $this->swish();
        $this->klarna();
        $this->vipps();
        $this->mobilePay();
        $this->gpp2();
        $this->plcc();
        $this->bankOp();
    }

    /**
     * @return array
     */
    public function toArray(): array
    {
        return $this->data;
    }

    /**
     * @return false|string
     */
    public function toJson(): string
    {
        return json_encode($this->data);
    }

    public function getCurrencyPrecision($currencyCode): int
    {
        $payment_currency_id = Currency::getIdByIsoCode($currencyCode);
        $payment_currency = Currency::getCurrencyInstance($payment_currency_id);

        return $payment_currency->precision;
    }

    /**
     * @return array
     */
    private function getLanguage(): array
    {
        $context = Context::getContext();
        $lang = $context->language->iso_code;
        $lang = (in_array($lang, ['nb', 'nn'])) ? 'no' : $lang;
        //check for dutch to so code
//        $lang = $lang === 'nl'? $lang = 'nl-be':$lang;....
        return ['default_language' => $lang];
    }

    /**
     * Card payments
     * @return void
     */
    private function card()
    {
        if ($this->settings->getCardEnabled()) {

            $this->data['configurations']['card'] = [
                'capture_now' => $this->settings->getCapture(),
                'payment_contract_id' => $this->settings->getPaymentContractId(),
                'authorization_type' => $this->settings->getTransactionType()
            ];
            if ($this->settings->getCardFraudProtectionEnabled()) {
                $this->data['configurations']['card']['fraud_protection_contract'] = $this->settings->getCardFraudProtectionContractId();
            }

            if ($this->settings->getThreeDSEnabled()) {
                $this->data['configurations']['card']['threed_secure'] = [
                    'threeds_contract_id' => $this->settings->getThreeDSContractId(),
                    'enabled' => true,
                    'transaction_mode' => $this->transactionType,
                ];
            }
            if ($this->settings->getAbsInstallmentsEnabled()) {
                $this->data['configurations']['card']['credit_term'] = 'STANDARD';
                $total = $this->helper->calculateAmount($this->cart->getOrderTotal(true, CartCore::BOTH), $this->currencyIso);
                $nrOfInstallments = $this->settings->getAbsInstallments($total);
                if ($nrOfInstallments) {
                    $this->data['configurations']['card']['credit_term'] = 'INSTALMENT_STANDARD';
                    $this->data['configurations']['card']['instalment'] = ['max_number_of_instalments' => $nrOfInstallments];
                }
            }
        }
    }

    /**
     * PLCC payments
     * @return void
     */
    private function plcc()
    {
        if ($this->settings->getPlccEnabled()) {
            $this->data['configurations']['plcc'] = [
                'capture_now' => $this->settings->getCapture(),
                'payment_contract_id' => $this->settings->getPlccContractId(),
                'authorization_type' => $this->settings->getTransactionType()
            ];
        }
    }
    /**
     * BankOp payments
     * @return void
     */
    private function bankOp()
    {
        if ($this->settings->getBankOpEnabled()) {
            $this->data['configurations']['bank'] = [
                'op' => ['payment_contract_id' => $this->settings->getBankOpContractId()],
                'authorization_type' => 'FINAL_AUTH'
            ];
        }
    }
    /**
     * PayPal payments
     * @return void
     */
    private function paypal()
    {
        if ($this->settings->getPaypalEnabled()) {
            $this->data['configurations']['paypal'] = [
                'capture_now' => $this->settings->getCapture(),
                'payment_contract_id' => $this->settings->getPaypalContractId(),
            ];
        }
    }

    /**
     * ThemeId - if not empty then we send it
     * @return void
     */
    private function theme()
    {
        if (!empty($this->settings->getThemeId())) {
            $this->data['theme_id'] = $this->settings->getThemeId();
        }
    }

    /**
     * ApplePay - APM
     * @return void
     */
    private function applePay()
    {
        if ($this->settings->getApplePayEnabled()) {
            $this->data['configurations']['apple_pay'] = [
                'capture_now' => $this->settings->getCapture(),
                'card' => [
                    'sca_compliance_level' => $this->settings->getApplePaySCALevel(),
                    'payment_contract_id' => $this->settings->getApplePayContractId(),
                ]
            ];

            if ($this->settings->getApplePaySCALevel() === 'FORCE_3DS') {
                $this->data['configurations']['apple_pay']['card']['threed_secure'] = [
                    'threeds_contract_id' => $this->settings->getApplePayThreeDSContractId(),
                    'transaction_mode' => $this->transactionType,
                ];
            }
        }
    }

    /**
     * GoolgePay - APM
     * @return void
     */
    private function googlePay()
    {
        if ($this->settings->getGooglePayEnabled()) {
            $this->data['configurations']['google_pay'] = [
                'capture_now' => $this->settings->getCapture(),
                'card' => [
                    'sca_compliance_level' => $this->settings->getGooglePaySCALevel(),
                    'payment_contract_id' => $this->settings->getGooglePayContractId(),
                ]
            ];

            if ($this->settings->getGooglePaySCALevel() !== 'NONE') {
                $this->data['configurations']['google_pay']['card']['threed_secure'] = [
                    'threeds_contract_id' => $this->settings->getGooglePayThreeDSContractId(),
                    'transaction_mode' => $this->transactionType,
                ];
            }
        }
    }

    /**
     * Klarna - APM
     * @return void
     */
    private function klarna()
    {
        if ($this->settings->getKlarnaEnabled()) {
            $this->data['configurations']['klarna'] = [
                'capture_now' => $this->settings->getCapture()
            ];

            $totalLineItemsAmount = 0;
            $products = $this->cart->getProducts();
            foreach ($products as $product) {
                $totalTaxAmount = $this->helper->calculateAmount($product['total_wt'], $this->currencyIso) - $this->helper->calculateAmount($product['total'], $this->currencyIso);
                $productAmount = $this->helper->calculateAmount($product['total_wt'], $this->currencyIso);
                $totalLineItemsAmount += $productAmount;
                $this->data['line_items'][] = [
                    'name' => $product['name'],
                    'quantity' => $product['quantity'],
                    'unit_price' => $this->helper->calculateAmount($product['price_wt'], $this->currencyIso),
                    'tax_rate' => $product['rate'],
                    'total_tax_amount' => $totalTaxAmount,
                    'total_amount' => $productAmount,
                ];
            }

            $appliedDiscounts = $this->cart->getCartRules();
            if (!empty($appliedDiscounts)) {
                $discountValue = 0;
                foreach ($appliedDiscounts as $discount) {
                    $discountValue += $discount['value_real'];
                }
                $totalWithoutDiscount = $this->cart->getOrderTotal(true, Cart::ONLY_PRODUCTS) - $discountValue;
                $total = $this->helper->calculateAmount($totalWithoutDiscount, $this->currencyIso);
                $totalLineItemsAmount = $total;
                $this->data['line_items'] = [
                    [
                        'name' => 'Cart ' . $this->cart->id,
                        'quantity' => 1,
                        'unit_price' => $total,
                        'tax_rate' => 0,
                        'total_tax_amount' => 0,
                        'total_amount' => $total,
                    ]
                ];
            }

            $carrier = new Carrier($this->cart->id_carrier);
            if (Validate::isLoadedObject($carrier)) {
                $shippingCost = $this->cart->getTotalShippingCost(null, true, null, $carrier->id);
                $shippingCostFormated = $this->helper->calculateAmount($shippingCost, $this->currencyIso);
                $shippingTax = $shippingCost - ($shippingCost / (1 + ($carrier->getTaxesRate() / 100)));
                $totalLineItemsAmount += $shippingCostFormated;

                if ($shippingCost > 0) {
                    $this->data['line_items'][] = [
                        'name' => $carrier->name,
                        'quantity' => 1,
                        'unit_price' => $shippingCostFormated,
                        'tax_rate' => $carrier->getTaxesRate(),
                        'total_tax_amount' => $this->helper->calculateAmount($shippingTax, $this->currencyIso),
                        'total_amount' => $shippingCostFormated,
                    ];
                }
            }

            //Final check of lineitems total amounts
            if ($totalLineItemsAmount !== $this->data['amount']) {
                $this->data['line_items'] = [
                    [
                        'name' => 'Cart ' . $this->cart->id,
                        'quantity' => 1,
                        'unit_price' => $this->data['amount'],
                        'tax_rate' => 0,
                        'total_tax_amount' => 0,
                        'total_amount' => $this->data['amount'],
                    ]
                ];
            }
        }
    }

    /**
     * Swish - APM
     * @return void
     */
    private function swish()
    {
        if ($this->settings->getSwishEnabled()) {
            $this->data['configurations']['swish'] = [
                'shopper_interaction' => $this->transactionType === 'P' ? 'MCOMMERCE' : 'ECOMMERCE'
            ];
        }
    }

    /**
     * Vipps - APM
     * @return void
     */
    private function vipps()
    {
        if ($this->settings->getVippsEnabled()) {
            $this->data['configurations']['vipps'] = [
                'capture_now' => $this->settings->getCapture(),
                'payment_contract_id' => $this->settings->getVippsContractId(),
                'card' => [
                    'sca_compliance_level' => $this->settings->getVippsSCALevel(),
                    'authorization_type' => $this->settings->getTransactionType()
                ]
            ];

            if ($this->settings->getVippsSCALevel() !== 'NONE') {
                $this->data['configurations']['vipps']['card']['threed_secure'] = [
                    'enabled' => true,
                    'threeds_contract_id' => $this->settings->getVippsThreeDSContractId(),
                    'transaction_mode' => $this->transactionType,
                ];
            }
        }
    }

    /**
     * GPP2 - APM
     * @return void
     */
    private function gpp2()
    {
        if ($this->settings->getGpp2Enabled()) {
            $this->data['configurations']['gpp2'] = [
                'payment_contract_id' => $this->settings->getGpp2ContractId(),
                'shopper_interaction' => 'ECOMMERCE'
            ];
        }
    }

    /**
     * MobilePay - APM
     * @return void
     */
    private function mobilePay()
    {
        if ($this->settings->getMobilePayEnabled()) {
            $this->data['configurations']['mobile_pay'] = [
                'capture_now' => $this->settings->getCapture(),
                'payment_contract_id' => $this->settings->getMobilePayContractId(),
                'card' => [
                    'sca_compliance_level' => $this->settings->getMobilePaySCALevel(),
                    'authorization_type' => $this->settings->getTransactionType(),
                ]
            ];
            if ($this->settings->getMobilePaySCALevel() !== 'NONE') {
                $this->data['configurations']['mobile_pay']['card']['threed_secure'] = [
                    'threeds_contract_id' => $this->settings->getMobilePayThreeDSContractId(),
                    'transaction_mode' => $this->transactionType,
                    'enabled' => true,
                ];
            }
        }
    }

    /**
     * Gift Card - Card option
     * @return void
     */
    private function giftCard()
    {
        if ($this->settings->getGiftCardEnabled()) {
            $this->data['configurations']['gift_card'] = [
                'capture_now' => $this->settings->getCapture(),
                'card' => [
                    'payment_contract_id' => $this->settings->getGiftCardContractId(),
                ]];
        }
    }
}
