<?php

use VerifoneEcomAPI\ApiWrapper\Settings;

/**
 * Class Verifone_Customer
 */
class Verifone_Customer extends Verifone_Api_Entity
{

    /**
     * @var \Order
     */
    private $cart, $address, $customer;

    /**
     * @var array
     */
    protected $data;

    /**
     * @var Settings
     */
    private $settings;

    /**
     * Verifone_Customer constructor.
     *
     * @param $params
     * @param Settings $settings
     */
    public function __construct(
        $params,
        Settings $settings
    )
    {
        $this->cart = $params['cart'];
        $this->address = $this->getAddresses();
        $this->customer = $params['customer'];
        $this->settings = $settings;
        $this->data = $this->buildData();
    }

    public function getAddresses()
    {
        $addressArr = [];
        if (isset($this->cart->id_address_invoice)) {
            $addressArr['invoice'] = new Address($this->cart->id_address_invoice);
        }

        if (isset($this->cart->id_address_delivery)) {
            $addressArr['delivery'] = new Address($this->cart->id_address_delivery);
        }

        return $addressArr;
    }

    /**
     * @return array
     */
    private function buildData()
    {
        $data = [];
        if (!empty($this->address['invoice']->company)) {
            $data['company_name'] = substr($this->address['invoice']->company, 0, 100);
        }
        $data['email_address'] = substr($this->customer->email, 0, 250);
        $data['entity_id'] = $this->settings->getEntityId();

        $data = array_merge($data, $this->getBillingDetails());

        return array_merge($data, $this->getShippingDetails());
    }

    /**
     * @return array
     */
    private function getShippingDetails()
    {
        // use billing details if no shipping details are found
        if (!isset($this->address['delivery'])) {
            $billingDetails = $this->createBillingDetails();
            if (!empty($billingDetails)) {
                return [
                    'shipping' => $billingDetails,
                ];
            } else {
                return [];
            }

        }

        $delivery = $this->address['delivery'];
        $shipping = [];

        if (!empty($delivery->address1)) {
            $shipping['address_1'] = substr($delivery->address1, 0, 40);
        }
        if (!empty($delivery->city)) {
            $shipping['city'] = substr($delivery->city, 0, 28);
        }
        if (!empty($delivery->id_country)) {
            $shipping['country_code'] = Country::getIsoById($delivery->id_country);
        }
        if (!empty($delivery->postcode)) {
            $shipping['postal_code'] = substr($delivery->postcode, 0, 10);
        }
        if (!empty($delivery->firstname)) {
            $shipping['first_name'] = substr($delivery->firstname, 0, 50);
        }
        if (!empty($delivery->lastname)) {
            $shipping['last_name'] = substr($delivery->lastname, 0, 50);
        }
        if (!empty($delivery->phone)) {
            $shipping['phone'] = preg_replace('/[^0-9]+/', '', substr($delivery->phone, 0, 20));
        }
        $state = $this->getState($delivery->id_state);
        if ($state) {
            $shipping['state'] = $state['iso_code'];
        }
        if ('' !== $delivery->address2) {
            $shipping['address_2'] = substr($delivery->address2, 0, 40);
        }

        if (!empty($shipping)) {
            return [
                'shipping' => $shipping,
            ];
        } else {
            return [];

        }
    }

    /**
     * @return array
     */
    private function createBillingDetails()
    {
        $invoice = $this->address['invoice'];

        $billing = [];
        if (!empty($invoice->address1)) {
            $billing['address_1'] = substr($invoice->address1, 0, 40);
        }
        if (!empty($invoice->city)) {
            $billing['city'] = substr($invoice->city, 0, 28);
        }
        if (!empty($invoice->id_country)) {
            $billing['country_code'] = strtoupper(Country::getIsoById($invoice->id_country));
        }
        if (!empty($invoice->firstname)) {
            $billing['first_name'] = substr($invoice->firstname, 0, 50);
        }
        if (!empty($invoice->lastname)) {
            $billing['last_name'] = substr($invoice->lastname, 0, 50);
        }
        if (!empty($invoice->postcode)) {
            $billing['postal_code'] = substr($invoice->postcode, 0, 10);
        }

        if (!empty($invoice->phone)) {
            $billing['phone'] = preg_replace('/[^0-9]+/', '', substr($invoice->phone, 0, 20));
        }
        $state = $this->getState($invoice->id_state);
        if ($state) {
            $billing['state'] = $state['iso_code'];
        }

        if ('' !== $invoice->address2) {
            $billing['address_2'] = substr($invoice->address2, 0, 40);
        }

        return $billing;
    }

    /**
     * @return array
     */
    private function getBillingDetails()
    {
        $billingDetails = $this->createBillingDetails();
        if (!empty($billingDetails)) {
            return [
                'billing' => $billingDetails,
            ];
        } else {
            return [];

        }
    }

    /**
     * @return array
     */
    public function toArray(): array
    {
        return $this->data;
    }

    /**
     * @return false|string
     */
    public function toJson(): string
    {
        return json_encode($this->data);
    }

    private function getState($id)
    {
        return Db::getInstance()->getRow(' SELECT * FROM `' . _DB_PREFIX_ . 'state` WHERE `id_state` = ' . pSQL($id), false);
    }

}
