<?php

require_once 'libs/interfaces/interface-verifone-hosted-arrayable.php';
require_once 'libs/interfaces/interface-verifone-hosted-jsonable.php';
require_once 'libs/abstract/abstract-verifone-api-entity.php';
require_once 'libs/includes/class-verifone-hosted-logger.php';
require_once 'libs/includes/class-verifone-hosted-checkout.php';
require_once 'libs/includes/class-verifone-hosted-helper.php';
require_once 'libs/includes/class-verifone-hosted-customer.php';
require_once 'libs/includes/class-verifone-hosted-helper.php';
require_once 'libs/includes/class-verifone-refund.php';
require_once 'libs/VerifoneEcomAPI/autoload.php';

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;
use VerifoneEcomAPI\ApiWrapper\Authentication\BasicAuth;
use VerifoneEcomAPI\ApiWrapper\Http\ApiClient\Client;
use VerifoneEcomAPI\ApiWrapper\Http\SimpleCurl;
use VerifoneEcomAPI\ApiWrapper\Regions\Au;
use VerifoneEcomAPI\ApiWrapper\Regions\Emea;
use VerifoneEcomAPI\ApiWrapper\Regions\Na;
use VerifoneEcomAPI\ApiWrapper\Regions\NzAu;
use VerifoneEcomAPI\ApiWrapper\Settings;


if (!defined('_PS_VERSION_')) {
    exit;
}

/**
 * Class verifone - safe payment method
 */
class Verifone extends PaymentModule
{
    const DEBUG_MODE = false;
    public $details;
    public $owner;
    public $name;
    public $address;
    public $bootstrap;
    public $is_eu_compatible;
    public $extra_mail_vars;
    public $confirmUninstall;
    protected $_html = '';
    protected $requiredFields = [
        'VERIFONE_API_KEY',
        'VERIFONE_USER_ID',
        'VERIFONE_ENTITY_ID',
        'VERIFONE_TITLE',
    ];

    protected $allFields = [
        'VERIFONE_MODE',
        'VERIFONE_API_KEY',
        'VERIFONE_USER_ID',
        'VERIFONE_ENTITY_ID',
        'VERIFONE_REGION',
        'VERIFONE_CHECKOUT_TYPE',
        'VERIFONE_TITLE',
        'VERIFONE_DESCRIPTION',
        'VERIFONE_THEME_ID',
        'VERIFONE_CARD_ENABLED',
        'VERIFONE_PAYMENT_CONTRACT_ID',
        'VERIFONE_3DS_ENABLED',
        'VERIFONE_3DS_CONTRACT_ID',
        'VERIFONE_PAYPAL_ENABLED',
        'VERIFONE_PAYPAL_CONTRACT_ID',
        'VERIFONE_APPLE_PAY_ENABLED',
        'VERIFONE_APPLE_PAY_CONTRACT_ID',
        'VERIFONE_APPLE_PAY_THREE_DS_CONTRACT_ID',
        'VERIFONE_APPLE_PAY_SCA_LEVEL',
        'VERIFONE_GOOGLE_PAY_ENABLED',
        'VERIFONE_GOOGLE_PAY_CONTRACT_ID',
        'VERIFONE_GOOGLE_PAY_THREE_DS_CONTRACT_ID',
        'VERIFONE_GOOGLE_PAY_SCA_LEVEL',
        'VERIFONE_KLARNA_ENABLED',
        'VERIFONE_CURL_VERIFY_SSL',
        'VERIFONE_TRANSACTION_TYPE',
        'VERIFONE_CARD_FRAUD_PROTECTION_ENABLED',
        'VERIFONE_CARD_FRAUD_PROTECTION_CONTRACT_ID',
        'VERIFONE_SWISH_ENABLED',
        'VERIFONE_VIPPS_ENABLED',
        'VERIFONE_VIPPS_CONTRACT_ID',
        'VERIFONE_VIPPS_THREE_DS_CONTRACT_ID',
        'VERIFONE_VIPPS_SCA_LEVEL',
        'VERIFONE_GPP2_ENABLED',
        'VERIFONE_GPP2_CONTRACT_ID',
        'VERIFONE_MOBILE_PAY_ENABLED',
        'VERIFONE_MOBILE_PAY_CONTRACT_ID',
        'VERIFONE_MOBILE_PAY_THREE_DS_CONTRACT_ID',
        'VERIFONE_MOBILE_PAY_SCA_LEVEL',
        'VERIFONE_ABS_INSTALLMENTS_ENABLED',
        'VERIFONE_ABS_INSTALLMENTS',
        'VERIFONE_ORDER_STATUS',
        'VERIFONE_GIFT_CARD_ENABLED',
        'VERIFONE_GIFT_CARD_CONTRACT_ID',
        'VERIFONE_PLCC_ENABLED',
        'VERIFONE_PLCC_CONTRACT_ID',
        'VERIFONE_BANK_OP_ENABLED',
        'VERIFONE_BANK_OP_CONTRACT_ID'
    ];

    /**
     * @var Client
     */
    public $client;

    /**
     * @var Verifone_Helper
     */
    public $verifone_helper;

    /**
     * @var Settings
     */
    public $verifone_settings;

    /**
     * @var Verifone_Cart_Logger
     */
    public $logger;

    /**
     * @var Emea|Na|NzAu|Au
     */
    public $mapper;

    /**
     * @var SimpleCurl
     */
    public $simple_curl;

    /**
     * @var BasicAuth
     */
    public $auth;

    /**
     * verifone constructor.
     */
    public function __construct()
    {
        $this->name = 'verifone';
        $this->tab = 'payments_gateways';
        $this->version = '3.2.4';
        $this->ps_versions_compliancy = ['min' => '1.7', 'max' => _PS_VERSION_];
        $this->author = 'Connectors Team - Verifone';
        $this->controllers = ['validation'];
        $this->is_eu_compatible = true;
        $this->currencies = true;
        $this->currencies_mode = 'checkbox';
        $this->verifone_settings = new Settings();
        $this->logger = new Verifone_Cart_Logger();
        $this->simple_curl = new SimpleCurl();

        // mini factory :D
        switch ($this->verifone_settings->getRegion()) {
            case 'AU':
                $this->mapper = new Au($this->verifone_settings);
                break;
            case 'NZAU':
                $this->mapper = new NzAu($this->verifone_settings);
                break;
            case 'NA':
                $this->mapper = new Na($this->verifone_settings);
                break;
            default:
                $this->mapper = new Emea($this->verifone_settings);
                break;
        }

        $this->auth = new BasicAuth($this->verifone_settings);
        $this->client = new Client($this->mapper, $this->verifone_settings, $this->auth, $this->simple_curl);

        $this->verifone_helper = new Verifone_Helper();
        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = 'Verifone';
        $this->description = $this->displayName . $this->l(' safe payment solutions');

        if (!count(Currency::checkPaymentCurrencies($this->id))) {
            $this->warning = $this->l('No currency has been set for this module.');
        }
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall Verifone payment modules?');
    }

    /**
     * install the module
     * @return bool
     */
    public function install(): bool
    {


        if (parent::install()
            && $this->registerHook('displayHeader')
            && $this->registerHook('paymentOptions')
            && $this->registerHook('displayBackOfficeHeader')
            && $this->registerHook('displayOrderConfirmation')
            && $this->registerHook('actionProductCancel')
            && $this->registerHook('actionOrderStatusUpdate')
            && $this->registerHook('displayPaymentTop')
            && $this->registerHook('actionOrderSlipAdd')
            && $this->registerHook('displayAdminOrderTop')
        ) {
            return true;
        }

        return false;
    }

    /**
     * uninstalls the module and deletes the config keys
     * @return bool
     * @throws PrestaShopException
     */
    function uninstall(): bool
    {

        $configs = Configuration::getMultiple($this->allFields);
        foreach ($configs as $key => $config) {
            Configuration::deleteByName($key);
        }

        return $this->unregisterHook('displayHeader')
            && $this->unregisterHook('paymentOptions')
            && $this->unregisterHook('displayBackOfficeHeader')
            && $this->unregisterHook('displayOrderConfirmation')
            && $this->unregisterHook('actionProductCancel')
            && $this->unregisterHook('displayShoppingCart')
            && $this->unregisterHook('displayPaymentTop')
            && $this->unregisterHook('actionOrderStatusUpdate')
            && $this->unregisterHook('actionOrderSlipAdd') //partial refunds
            && $this->unregisterHook('displayAdminOrderTop')
            && parent::uninstall();
    }

    public function hookDisplayBackOfficeHeader()
    {

        $this->context->controller->addJS($this->_path . '/views/assets/js/setup.js');
    }

    public function hookActionOrderSlipAdd($params)
    {
        $refundAmount = 0;
        foreach ($params['productList'] as $product) {
            $refundAmount = $refundAmount + $product['amount'];
        }
    }

    public function hookDisplayPaymentTop()
    {
        $error = $this->context->cookie->__get('redirect_error');
        $this->context->smarty->assign('verifone_error', $error);
        $this->context->cookie->__unset('redirect_error');
        $this->context->cookie->write();

        return $this->display(__FILE__, 'views/templates/hook/cart_error.tpl');
    }

    /**
     * show the settings page, also saves and validates the form on submit
     * @return string
     * @throws SmartyException
     * @throws PrestaShopException
     */
    public function getContent(): string
    {
        $output = '';
        // default/saved data
        $token = Tools::getAdminTokenLite('AdminModules');
        $lang = Configuration::get('PS_LANG_DEFAULT');
        $this->context->smarty->assign([
            'data' => Configuration::getMultiple($this->allFields),
            'action' => AdminController::$currentIndex . '&configure=' . $this->name . '&token=' . $token,
            'webhookUrl' => $this->context->link->getModuleLink($this->name, 'webhook', [], true),
            'defaultOrderStatus' => Configuration::get('PS_OS_PAYMENT'),
            'prestashopOrderStatues' => (new OrderState($lang))->getOrderStates($lang),
            'autoSetupUrl' => $this->context->link->getModuleLink($this->name, 'autoSetup', [], true),
        ]);

        if (Tools::isSubmit('submit' . $this->name)) {
            $this->reset_token();
            $postData = Tools::getAllValues();

            foreach ($postData as $key => $value) {
                if (strpos($key, 'VERIFONE_') !== false) {
                    if (strpos($key, '_ID') !== false) { // must be a valid UUID
                        $this->verifone_helper->uuid($key, $value);
                    }
                    if (in_array($key, $this->requiredFields)) { // mandatory fields
                        $this->verifone_helper->required($key, $value);
                    }
                }
                if (!empty($postData['VERIFONE_ABS_INSTALLMENTS'])) {

                    $this->verifone_helper->checkInstallments($postData['VERIFONE_ABS_INSTALLMENTS']);
                }
            }

            //save new data if ok
            if ($this->verifone_helper->hasErrors() === false) {
                // remove old data
                $configs = Configuration::getMultiple($this->allFields);
                foreach ($configs as $key => $config) {
                    Configuration::deleteByName($key);
                }
                foreach ($this->allFields as $key) {
                    if (is_array($postData[$key])) {
                        $postData[$key] = json_encode($postData[$key]);
                    }
                    Configuration::updateValue($key, $postData[$key]);
                }
                $output .= $this->displayConfirmation($this->l('Settings updated!'));
            } else {
                $this->context->smarty->assign('vf_errors', $this->verifone_helper->getErrorBag());
                $output .= $this->displayError($this->l('Please fix all the errors before you save!'));
            }
            //overwrite old data with the new data
            $this->context->smarty->assign('data', $postData);
        }
        $this->context->smarty->assign(['installments' => json_decode(Configuration::get('VERIFONE_ABS_INSTALLMENTS'), true)]);

        $output .= $this->context->smarty->fetch('module:verifone/views/templates/back/admin_tabs.tpl');

        return $output;
    }

    /**
     * ==============================HOOKS==============================
     * @throws PrestaShopDatabaseException
     */

    public function hookHeader(): ?string
    {
        if (Tools::getValue('controller') == "order" && $this->checkActiveModule()) {
            Media::addJsDef(['verifone_verify_url' => $this->context->link->getModuleLink($this->name, 'generateUrl', ['module_name' => $this->name], true)]);
            $this->context->controller->registerJavascript($this->name . '-gen-url-context', 'modules/' . $this->name . '/views/assets/js/validate.js');
            $this->context->controller->registerJavascript($this->name . '-gen-url-context-iframe', 'modules/' . $this->name . '/views/assets/js/iframe.js');
        }

        return null;
    }

    /**
     * @param $params
     *
     * @return null[]|PaymentOption[]|void
     */
    public function hookPaymentOptions($params)
    {
        if (!$this->active || !$this->checkCurrency($params['cart'])) {
            return;
        }
        // we clear the cache for every change we make
        Tools::clearSmartyCache();
        Tools::clearXMLCache();
        Media::clearCache();
        Tools::generateIndex();

        return [$this->getVerifonePaymentOption()];
    }

    /**
     * @param $params
     *
     * @return false|string
     */
    public function hookDisplayOrderConfirmation($params)
    {
        if ((!isset($params['order']) || $params['order']->module != $this->name) || !$this->active) {
            return false;
        }
        $cart = $params['cart'] ?? null;
        $this->smarty->assign(['order' => $params['order'], 'cart' => $cart]);

        return $this->fetch('module:verifone/views/templates/hook/payment_return.tpl');
    }

    /**
     * @throws PrestaShopDatabaseException
     * @throws PrestaShopException
     */
    public function hookDisplayAdminOrderTop($params)
    {

        $id_order = $params['id_order'] ?? null;
        $order = new Order((int)$id_order);
        if (Validate::isLoadedObject($order) && $order->module === $this->name) {
            $orderPayment = OrderPayment::getByOrderReference($order->reference);
            if (isset($orderPayment[0])) {
                try {
                    $transaction = $this->client->getTransaction($orderPayment[0]->transaction_id);

                    return $this->manualCaptureButton($order, $transaction);
                } catch (Exception $exception) {
                    //no action
                }
            }
        }
        return false;
    }

    /** partial REFUND
     *
     * @param $params
     * @throws Exception
     */
    public function hookActionProductCancel($params)
    {
        $order = $params['order'] ?? null;
        if (Validate::isLoadedObject($order) && $order->module === $this->name) {
            $this->addOrderNote($order, 'Partial refunds are not supported! You need to login to Verifone Central and issue the refund from there too.');
        }
    }

    /**
     * =============================END HOOKS==============================
     * @throws SmartyException
     */

    /**
     * allow capture for transaction_type = 'AUTHORISATION'/'PREAUTH'
     *
     * @param $order
     * @param $transaction
     * @return false|string|null
     * @throws SmartyException
     */
    private function manualCaptureButton($order, $transaction)
    {
        if (isset($transaction['status']) &&
            in_array($transaction['status'], ['PREAUTH AUTHORISED', 'AUTHORISATION AUTHORISED', 'AUTHORISED']) &&
            in_array($transaction['transaction_type'], ['AUTHORISATION', 'PREAUTH'])) {
            $this->context->smarty->assign([
                'transaction_id' => $transaction['id'],
                'transaction' => $transaction,
                'order' => $order,
                'captureUrl' => $this->context->link->getModuleLink($this->name, 'capture', [], true),
            ]);

            return $this->context->smarty->fetch('module:verifone/views/templates/hook/manualCapture.tpl');
        }


        return null;
    }


    /**
     * @param $cart
     *
     * @return bool
     */
    public function checkCurrency($cart): bool
    {
        $currency_order = new Currency($cart->id_currency);
        $currencies_module = $this->getCurrency($cart->id_currency);

        if (is_array($currencies_module)) {
            foreach ($currencies_module as $currency_module) {
                if ($currency_order->id == $currency_module['id_currency']) {
                    return true;
                }
            }
        }

        return false;
    }

    public function getVerifonePaymentOption(): ?PaymentOption
    {

        $protocol = $this->isSSL() ? _PS_BASE_URL_SSL_ : _PS_BASE_URL_;
        $this->smarty->assign([
            'description' => $this->verifone_settings->getDescription(),
            'logo' => $protocol . __PS_BASE_URI__ . 'modules/' . $this->name . '/logo.svg'
        ]);

        $option = new PaymentOption();
        $option->setCallToActionText($this->verifone_settings->getTitle());

        if ($this->verifone_settings->getCheckoutType() === 'hpp') {
            $option->setAdditionalInformation($this->fetch('module:verifone/views/templates/hook/description.tpl'))
                ->setAction('javascript:VerifoneCheckout()');
        } else {
            $option->setAdditionalInformation($this->fetch('module:verifone/views/templates/hook/iframe.tpl'))
                ->setBinary(true);
        }

        return $option;
    }

    /**
     * generates the form for the payment option (hosted & inline)
     * @return string
     * @throws SmartyException
     */
    protected function generateForm(): string
    {
        return $this->context->smarty->fetch('module:verifone/views/templates/hook/payment_options.tpl');
    }

    /**
     * Resets the token on ANY setting change
     */
    public function reset_token()
    {
        if (false !== Configuration::get('__verifone_api_token')) {
            Configuration::deleteByName('__verifone_api_token');
        }
    }

    /**
     * @throws PrestaShopDatabaseException
     */
    public function checkActiveModule(): bool
    {
        $active = false;
        $modules = Hook::getHookModuleExecList('paymentOptions') ?? [];
        foreach ($modules as $module) {
            if ($module['module'] == $this->name) {
                $active = true;
                break;
            }
        }

        return $active;
    }

    public function hasPaidShipping($shippingArr): bool
    {
        $total = 0;
        foreach ($shippingArr as $valuesArr) {
            $total += $valuesArr['shipping_cost_tax_incl'];
        }

        return $total > 0;
    }


    /**
     * check if it has SSL
     * @return bool
     */
    private function isSSL(): bool
    {
        return isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ||
            isset($_SERVER['REQUEST_SCHEME']) && $_SERVER['REQUEST_SCHEME'] === 'https' ||
            isset($_SERVER['HTTP_X_FORWARDED_PROTO']) && $_SERVER['HTTP_X_FORWARDED_PROTO'] == 'https';

    }

    /**
     *  For debuting purposes
     *
     * @param $data
     * @param bool $stop
     *
     * @return void
     */
    function dump($data, bool $stop = true)
    {
        highlight_string("<?php \n" . var_export($data, true) . ";\n?>");
        if ($stop) {
            exit;
        }
    }

    /**
     * @throws PrestaShopException
     * @throws PrestaShopDatabaseException
     * @throws Exception
     */
    public function hookActionOrderStatusUpdate($params)
    {
        // try to refund bulk orders
        if ($params['newOrderStatus']->id === (int)Configuration::get('PS_OS_REFUND')) {
            $order = new Order($params['id_order']);
            $this->verifoneRefund($order);
        }
        // try manual capture bulk order
        if ($params['newOrderStatus']->id === (int)Configuration::get('PS_OS_PAYMENT')) {
            $this->captureTransaction($params['id_order']);
        }
    }

    /**
     * only for certain flow: order with status 'processing in progress' -> 'payment accepted' and paid with verifone
     *
     * @param $orderId
     *
     * @return void
     */
    private function captureTransaction($orderId)
    {
        try {
            $order = new Order($orderId);

            if ($order->getCurrentOrderState()->id === (int)Configuration::get('PS_OS_PREPARATION') &&
                $order->getOrderPayments()[0]->payment_method === $this->name) {
                $transactionId = $order->getOrderPayments()[0]->transaction_id;
                // first, we fetch the transaction and check status
                $transaction = $this->client->getTransaction($transactionId);
                if (isset($transaction['status']) &&
                    in_array($transaction['status'], ['PREAUTH AUTHORISED', 'AUTHORISATION AUTHORISED', 'AUTHORISED']) &&
                    in_array($transaction['transaction_type'], ['AUTHORISATION', 'PREAUTH'])) {
                    //second, we perform the capture action
                    if ($transaction['payment_product'] === 'PayPal') {
                        $payload = [
                            'amount' => [
                                'value' => (int)(100 * $order->total_paid),
                                'currencyCode' => $transaction['currency_code']
                            ]
                        ];
                    } else {
                        $payload = ['amount' => (int)(100 * $order->total_paid)];
                    }

                    $response = $this->client->postCapture($payload, $transactionId, $transaction['payment_product']);
                    // if not success, we move the order into ERROR state
                    if (!isset($response['status']) && !in_array($response['status'], ['AUTHORISED', 'AUTHORIZED'])) {
                        $history = new OrderHistory();
                        $history->id_order = (int)$order->id;
                        $history->changeIdOrderState((int)Configuration::get('PS_OS_ERROR'), $order, true);
                        $history->save();
                    }
                }
            }
        } catch (Exception $exception) {
            // no action
        }
    }

    /**
     * @param $order
     * @throws Exception
     */
    private function verifoneRefund($order)
    {
        try {
            if (Validate::isLoadedObject($order) && $order->module === $this->name &&
                $order->getCurrentOrderState()->id != Configuration::get('PS_OS_REFUND') &&
                $this->preventDuplicateRefundRequests($order->id)) {
                $orderPayment = OrderPayment::getByOrderReference($order->reference);
                $transaction_id = null;
                foreach ($orderPayment as $payment) {
                    $invoice = $payment->getOrderInvoice($order->id);
                    if ($invoice) {
                        $transaction_id = trim($payment->transaction_id);
                        break;
                    }
                }
                if (!$transaction_id) {
                    $this->logger->error($this->l('Cannot find a valid transaction ID for this order'));
                    $this->context->controller->errors[] = $this->l('Payment module encountered an error. Cannot get transaction id.');
                }
                $refund = new Verifone_Refund($order->total_paid, $this->context->currency->iso_code, $this->verifone_helper);
                if ($refund->toArray()['amount'] >= 1) {
                    $response = $this->client->postRefund($refund->toArray(), $transaction_id);
                    if (!isset($response['id']) || !isset($response['status']) || 'REFUNDED' !== $response['status']) {
                        $error_msg = sprintf($this->l('Error Refunding Invoice for order "%s" with transaction id "%s". Response: %s'),
                            $order->reference, $transaction_id, print_r($response, true));
                        $this->logger->error($error_msg);
                        $this->context->controller->errors[] = $error_msg;
                        $this->addOrderNote($order, 'Unable to refund transaction! Make sure the transaction was SETTLED/CAPTURED 
                    in Verifone Portal and the refund amount is correct or a refund was already made.');
                        throw new Exception($error_msg);
                    } else {
                        $this->addOrderNote($order, 'Verifone full amount refunded with success');
                        $history = new OrderHistory();
                        $history->id_order = (int)$this->order->id;
                        $history->changeIdOrderState((int)Configuration::get('PS_OS_REFUND'), $order, true);
                        $history->save();
                    }
                }
            }

        } catch (Exception $e) {
            //do nothing... if we trow exception, the refund will not be marked as expect
        }

    }

    public function addOrderNote($order, $note): bool
    {
        try {
            $admin = Db::getInstance()->getRow('SELECT id_employee from ' . _DB_PREFIX_ . 'employee ORDER BY id_employee ASC');
            $thread = Db::getInstance()->getRow('SELECT id_customer_thread from ' . _DB_PREFIX_ . 'customer_thread WHERE id_order = ' . $order->id . ' AND id_shop =' . $this->context->shop->id);
            if ($thread) {
                $threadId = $thread['id_customer_thread'];
            } else {
                $now = date('Y-m-d H:i:s');
                $cnt = $this->context;

                $thread = new CustomerThread();
                $thread->id_shop = $cnt->shop->id;
                $thread->id_lang = $cnt->language->id;
                $thread->id_contact = 0;
                $thread->id_customer = $order->id_customer;
                $thread->id_order = $order->id;
                $thread->id_product = 0;
                $thread->status = 'open';
                $thread->email = $order->getCustomer()->email;
                $thread->date_add = $now;
                $thread->token = substr(str_shuffle("0123456789abcdefghijklmnopqrstvwxyzQWERTYUIOPLKJHGFDSAZXCVBNM"), 0, 12);;
                $thread->date_upd = $now;

                $thread->save();
                $threadId = $thread->id;
            }

            $message = new CustomerMessage();
            $message->id_order = $order->id;
            $message->id_employee = $admin['id_employee'];
            $message->id_customer_thread = $threadId;
            $message->message = $note;
            $message->private = true;

            return $message->save();

        } catch (Exception $exception) {
            $this->logger->error($exception->getMessage());
            return false;
        }
    }

    /**
     * @param $orderId
     * @return bool
     */
    public function preventDuplicateRefundRequests($orderId): bool
    {
        $result = true;
        $messages = CustomerMessage::getMessagesByOrderId($orderId);
        foreach ($messages as $message) {
            if ($message['message'] === 'Verifone full amount refunded with success') {
                $result = false;
            }
        }
        return $result;
    }

}
